// bangla-converter.js
// Full two-way converter: Unicode <-> Bijoy / SomewhereIn / Boisakhi

// ---------------------- Base functions ----------------------

function Insert(field, text) {
    if (document.selection) {
        field.focus();
        sel = document.selection.createRange();
        sel.text = text;
        sel.collapse(true);
        sel.select();
    } else if (field.selectionStart || field.selectionStart == '0') {
        var startPos = field.selectionStart;
        var endPos = field.selectionEnd;
        var scrollTop = field.scrollTop;
        startPos = (startPos == -1 ? field.value.length : startPos);
        field.value = field.value.substring(0, startPos) + text + field.value.substring(endPos, field.value.length);
        field.focus();
        field.selectionStart = startPos + text.length;
        field.selectionEnd = startPos + text.length;
        field.scrollTop = scrollTop;
    } else {
        var scrollTop = field.scrollTop;
        field.value += text;
        field.focus();
        field.scrollTop = scrollTop;
    }
}

function RemoveNInsert(field, value, len) {
    if (document.selection) {
        field.focus();
        sel = document.selection.createRange();
        if (field.value.length >= len) { sel.moveStart('character', -1 * (len)); }
        sel.text = value; sel.collapse(true); sel.select();
    }
    else if (field.selectionStart || field.selectionStart == 0) {
        field.focus();
        var startPos = field.selectionStart - len;
        var endPos = field.selectionEnd;
        var scrollTop = field.scrollTop;
        startPos = (startPos == -1 ? field.value.length : startPos);
        field.value = field.value.substring(0, startPos) + value + field.value.substring(endPos, field.value.length);
        field.focus();
        field.selectionStart = startPos + value.length;
        field.selectionEnd = startPos + value.length;
        field.scrollTop = scrollTop;
    } else {
        var scrollTop = field.scrollTop;
        field.value += value;
        field.focus();
        field.scrollTop = scrollTop;
    }
}

function capsDetect(e) {
    if (!e) e = window.event; if (!e) return false;
    var theKey = e.which ? e.which : (e.keyCode ? e.keyCode : (e.charCode ? e.charCode : 0));
    var theShift = e.shiftKey || (e.modifiers && (e.modifiers & 4));
    return ((theKey > 64 && theKey < 91 && !theShift) || (theKey > 96 && theKey < 123 && theShift));
}

function HideDIV(id) {
    if (document.getElementById) { document.getElementById(id).style.display = 'none'; }
    else {
        if (document.layers) { document.id.display = 'none'; }
        else { document.all.id.style.display = 'none'; }
    }
}
function ShowDIV(id) {
    if (document.getElementById) { document.getElementById(id).style.display = 'block'; }
    else {
        if (document.layers) { document.id.display = 'block'; }
        else { document.all.id.style.display = 'block'; }
    }
}

// ---------------------- Bangla checks & helpers ----------------------

function IsBanglaDigit(CUni) { return '০১২৩৪৫৬৭৮৯'.indexOf(CUni) >= 0; }
function IsBanglaPreKar(CUni) { return 'িৈে'.indexOf(CUni) >= 0; }
function IsBanglaPostKar(CUni) { return 'াোৌৗুূীৃ'.indexOf(CUni) >= 0; }
function IsBanglaKar(CUni) { return IsBanglaPreKar(CUni) || IsBanglaPostKar(CUni); }
function IsBanglaBanjonborno(CUni) {
    return 'কখগঘঙচছজঝঞটঠডঢণতথদধনপফবভমশষসহযরলয়ংঃঁৎ'.indexOf(CUni) >= 0;
}
function IsBanglaSoroborno(CUni) { return 'অআইঈউঊঋঌএঐওঔ'.indexOf(CUni) >= 0; }
function IsBanglaNukta(CUni) { return 'ংঃঁ'.indexOf(CUni) >= 0; }
function IsBanglaFola(CUni) { return (CUni == "্য" || CUni == "্র"); }
function IsBanglaHalant(CUni) { return CUni == '্'; }
function IsSpace(C) { return (C == ' ' || C == '\t' || C == '\n' || C == '\r'); }
function MapKarToSorborno(CUni) {
    var CSorborno = CUni;
    if (CUni == 'া') CSorborno = 'আ';
    else if (CUni == 'ি') CSorborno = 'ই';
    else if (CUni == 'ী') CSorborno = 'ঈ';
    else if (CUni == 'ু') CSorborno = 'উ';
    else if (CUni == 'ূ') CSorborno = 'ঊ';
    else if (CUni == 'ৃ') CSorborno = 'ঋ';
    else if (CUni == 'ে') CSorborno = 'এ';
    else if (CUni == 'ৈ') CSorborno = 'ঐ';
    else if (CUni == 'ো') CSorborno = 'ও';
    else if (CUni == "ো") CSorborno = 'ও';
    else if (CUni == 'ৌ') CSorborno = 'ঔ';
    else if (CUni == "ৌ") CSorborno = 'ঔ';
    return CSorborno;
}
function MapSorbornoToKar(CUni) {
    var CKar = CUni;
    if (CUni == 'আ') CKar = 'া';
    else if (CUni == 'ই') CKar = 'ি';
    else if (CUni == 'ঈ') CKar = 'ী';
    else if (CUni == 'উ') CKar = 'ু';
    else if (CUni == 'ঊ') CKar = 'ূ';
    else if (CUni == 'ঋ') CKar = 'ৃ';
    else if (CUni == 'এ') CKar = 'ে';
    else if (CUni == 'ঐ') CKar = 'ৈ';
    else if (CUni == 'ও') CKar = 'ো';
    else if (CUni == 'ঔ') CKar = 'ৌ';
    return CKar;
}

// ---------------------- bijoy / somewherein / boisakhi (ASCII -> Unicode) maps ----------------------
// (from your provided "bijoy file" content)
var bijoy_string_conversion_map={
"i¨":"র‌্য","ª¨":"্র্য","°":"ক্ক","±":"ক্ট","³":"ক্ত","K¡":"ক্ব","¯Œ":"স্ক্র","µ":"ক্র","K¬":"ক্ল","¶":"ক্ষ","ÿ":"ক্ষ","·":"ক্স","¸":"গু","»":"গ্ধ","Mœ":"গ্ন","M¥":"গ্ম","M­":"গ্ল","¼":"ঙ্ক","•¶":"ঙ্ক্ষ","•L":"ঙ্খ","½":"ঙ্গ","•N":"ঙ্ঘ","•":"ক্স","”P":"চ্চ","”Q":"চ্ছ","”Q¡":"চ্ছ্ব","”T":"চ্ঞ","¾¡":"জ্জ্ব","¾":"জ্জ","À":"জ্ঝ","Á":"জ্ঞ","R¡":"জ্ব","Â":"ঞ্চ","Ã":"ঞ্ছ","Ä":"ঞ্জ","Å":"ঞ্ঝ","Æ":"ট্ট","U¡":"ট্ব","U¥":"ট্ম","Ç":"ড্ড","È":"ণ্ট","É":"ণ্ঠ","Ý":"ন্স","Ê":"ণ্ড","š‘":"ন্তু","Y\\^":"ণ্ব","Ë":"ত্ত","Ë¡":"ত্ত্ব","Ì":"ত্থ","Z¥":"ত্ম","š—¡":"ন্ত্ব","Z¡":"ত্ব","Î":"ত্র","_¡":"থ্ব","˜M":"দ্গ","˜N":"দ্ঘ","Ï":"দ্দ","×":"দ্ধ","˜¡":"দ্ব","Ø":"দ্ব","™¢":"দ্ভ","Ù":"দ্ম","`ª“":"দ্রু","aŸ":"ধ্ব","a¥":"ধ্ম","›U":"ন্ট","Ú":"ন্ঠ","Û":"ন্ড","šÍ":"ন্ত","š—":"ন্ত","š¿":"ন্ত্র","š’":"ন্থ","›`":"ন্দ","›Ø":"ন্দ্ব","Ü":"ন্ধ","bœ":"ন্ন","š\\^":"ন্ব","b¥":"ন্ম","Þ":"প্ট","ß":"প্ত","cœ":"প্ন","à":"প্প","cø":"প্ল","c­":"প্ল","á":"প্স","d¬":"ফ্ল","â":"ব্জ","ã":"ব্দ","ä":"ব্ধ","eŸ":"ব্ব","e­":"ব্ল","å":"ভ্র","gœ":"ম্ন","¤ú":"ম্প","ç":"ম্ফ","¤\\^":"ম্ব","¤¢":"ম্ভ","¤£":"ম্ভ্র","¤§":"ম্ম","¤­":"ম্ল","i“":"রু","iæ":"রু","iƒ":"রূ","é":"ল্ক","ê":"ল্গ","ë":"ল্ট","ì":"ল্ড","í":"ল্প","î":"ল্ফ","j¦":"ল্ব","j¥":"ল্ম","jø":"ল্ল","ï":"শু","ð":"শ্চ","kœ":"শ্ন","kø":"শ্ল","k¦":"শ্ব","k¥":"শ্ম","k­":"শ্ল","®‹":"ষ্ক","®Œ":"ষ্ক্র","ó":"ষ্ট","ô":"ষ্ঠ","ò":"ষ্ণ","®ú":"ষ্প","õ":"ষ্ফ","®§":"ষ্ম","¯‹":"স্ক","÷":"স্ট","ö":"স্খ","¯—":"স্ত","¯Í":"স্ত","¯‘":"স্তু","¯¿":"স্ত্র","¯’":"স্থ","mœ":"স্ন","¯ú":"স্প","ù":"স্ফ","¯\\^":"স্ব","¯§":"স্ম","¯­":"স্ল","û":"হু","nè":"হ্ণ","ý":"হ্ন","þ":"হ্ম","n¬":"হ্ল","ü":"হৃ","©":"র্","Av":"আ","A":"অ","B":"ই","C":"ঈ","D":"উ","E":"ঊ","F":"ঋ","G":"এ","H":"ঐ","I":"ও","J":"ঔ","K":"ক","L":"খ","M":"গ","N":"ঘ","O":"ঙ","P":"চ","Q":"ছ","R":"জ","S":"ঝ","T":"ঞ","U":"ট","V":"ঠ","W":"ড","X":"ঢ","Y":"ণ","Z":"ত","_":"থ","`":"দ","a":"ধ","b":"ন","c":"প","d":"ফ","e":"ব","f":"ভ","g":"ম","h":"য","i":"র","j":"ল","k":"শ","l":"ষ","m":"স","n":"হ","o":"ড়","p":"ঢ়","q":"য়","r":"ৎ","0":"০","1":"১","2":"২","3":"৩","4":"৪","5":"৫","6":"৬","7":"৭","8":"৮","9":"৯","v":"া","w":"ি","x":"ী","y":"ু","z":"ু","~":"ূ","„":"ৃ","‡":"ে","†":"ে","‰":"ৈ","\\ˆ":"ৈ","Š":"ৗ","Ô":"‘","Õ":"’","\\|":"।","Ò":"“","Ó":"”","s":"ং","t":"ঃ","u":"ঁ","ª":"্র","Ö":"্র","«":"্র","¨":"্য","\\&":"্","…":"ৃ"
};

var somewherein_string_conversion_map={
"ª¨":"্র্য","i¨":"র‌্য","°":"ক্ক","±":"ক্ট","³":"ক্ত","K¡":"ক্ব","¯Œ":"স্ক্র","µ":"ক্র","K¬":"ক্ল","¶":"ক্ষ","·":"ক্স","¸":"গু","»":"গ্ধ","Mœ":"গ্ন","M¥":"গ্ম","M­":"গ্ল","¼":"ঙ্ক","•¶":"ঙ্ক্ষ","•L":"ঙ্খ","½":"ঙ্গ","•N":"ঙ্ঘ","•":"ক্স","”P":"চ্চ","”Q":"চ্ছ","”Q¡":"চ্ছ্ব","”T":"চ্ঞ","¾¡":"জ্জ্ব","¾":"জ্জ","À":"জ্ঝ","Á":"জ্ঞ","R¡":"জ্ব","Â":"ঞ্চ","Ã":"ঞ্ছ","Ä":"ঞ্জ","Å":"ঞ্ঝ","Æ":"ট্ট","U¡":"ট্ব","U¥":"ট্ম","Ç":"ড্ড","È":"ণ্ট","É":"ণ্ঠ","Ý":"ন্স","Ê":"ণ্ড","š‘":"ন্তু","Y\\^":"ণ্ব","Ë":"ত্ত","Ë¡":"ত্ত্ব","Ì":"ত্থ","Z¥":"ত্ম","š—¡":"ন্ত্ব","Z¡":"ত্ব","Î":"ত্র","_¡":"থ্ব","˜M":"দ্গ","˜N":"দ্ঘ","Ï":"দ্দ","×":"দ্ধ","˜¡":"দ্ব","Ø":"দ্ব","™¢":"দ্ভ","Ù":"দ্ম","`ª“":"দ্রু","aŸ":"ধ্ব","a¥":"ধ্ম","›U":"ন্ট","Ú":"ন্ঠ","Û":"ন্ড","šÍ":"ন্ত","š—":"ন্ত","š¿":"ন্ত্র","š’":"ন্থ","›`":"ন্দ","›Ø":"ন্দ্ব","Ü":"ন্ধ","bœ":"ন্ন","š\\^":"ন্ব","b¥":"ন্ম","Þ":"প্ট","ß":"প্ত","cœ":"প্ন","à":"প্প","cø":"প্ল","c­":"প্ল","á":"প্স","d¬":"ফ্ল","â":"ব্জ","ã":"ব্দ","ä":"ব্ধ","eŸ":"ব্ব","e­":"ব্ল","å":"ভ্র","gœ":"ম্ন","¤ú":"ম্প","ç":"ম্ফ","¤\\^":"ম্ব","¤¢":"ম্ভ","¤£":"ম্ভ্র","¤§":"ম্ম","¤­":"ম্ল","i“":"রু","iæ":"রু","iƒ":"রূ","é":"ল্ক","ê":"ল্গ","ë":"ল্ট","ì":"ল্ড","í":"ল্প","î":"ল্ফ","j¦":"ল্ব","j¥":"ল্ম","j­":"ল্ল","ï":"শু","ð":"শ্চ","kœ":"শ্ন","kø":"শ্ল","k¦":"শ্ব","k¥":"শ্ম","k­":"শ্ল","®‹":"ষ্ক","®Œ":"ষ্ক্র","ó":"ষ্ট","ô":"ষ্ঠ","ò":"ষ্ণ","®ú":"ষ্প","õ":"ষ্ফ","®§":"ষ্ম","¯‹":"স্ক","÷":"স্ট","ö":"স্খ","¯—":"স্ত","¯‘":"স্তু","¯¿":"স্ত্র","¯’":"স্থ","mœ":"স্ন","¯ú":"স্প","ù":"স্ফ","¯\\^":"স্ব","¯§":"স্ম","¯­":"স্ল","nè":"হ্ণ","ý":"হ্ন","þ":"হ্ম","û":"হু","n¬":"হ্ল","ü":"হৃ","©":"র্","Av":"আ","A":"অ","B":"ই","C":"ঈ","D":"উ","E":"ঊ","F":"ঋ","G":"এ","H":"ঐ","I":"ও","J":"ঔ","K":"ক","L":"খ","M":"গ","N":"ঘ","O":"ঙ","P":"চ","Q":"ছ","R":"জ","S":"ঝ","T":"ঞ","U":"ট","V":"ঠ","W":"ড","X":"ঢ","Y":"ণ","Z":"ত","_":"থ","`":"দ","a":"ধ","b":"ন","c":"প","d":"ফ","e":"ব","f":"ভ","g":"ম","h":"য","i":"র","j":"ল","k":"শ","l":"ষ","m":"স","n":"হ","o":"ড়","p":"ঢ়","q":"য়","r":"ৎ","0":"০","1":"১","2":"২","3":"৩","4":"৪","5":"৫","6":"৬","7":"৭","8":"৮","9":"৯","v":"া","w":"ি","x":"ী","y":"ু","z":"ু","~":"ূ","„":"ৃ","‡":"ে","†":"ে","‰":"ৈ","\\ˆ":"ৈ","Š":"ৗ","s":"ং","t":"ঃ","u":"ঁ","ª":"্র","Ö":"্র","«":"্র","¨":"্য","…":"ৃ","Ô":"‘","Õ":"’","\\|":"।","\\&":"্","Ò":"“","Ó":"”"
};

var boisakhi_string_conversion_map={
"Ûø":"্র্য","kø":"র‌্য","~":"ক্ক","ƒ":"ক্ট","£ß":"ক্ত","Kò":"ক্ব","Ç":"স্ক্র","¢ß":"ক্র","qô":"ক্ষ্ম","q":"ক্ষ","…":"ক্স","•":"ক্স","Kõ":"ক্ল","†":"গু","‡":"গ্গ","ˆ":"গ্ধ","Mí":"গ্ন","Mô":"গ্ম","Mö":"গ্ল","‰":"ঙ্ক","áq":"ঙ্ক্ষ","áL":"ঙ্খ","Š":"ঙ্গ","áN":"ঙ্ঘ","âP":"চ্চ","âQ":"চ্ছ","âQò":"চ্ছ্ব","Œò":"জ্জ্ব","Œ":"জ্জ","š":"জ্ঞ","Rò":"জ্ব","é":"ঞ্চ","˜":"ঞ্ছ","™":"ঞ্জ","ã":"ঞ্ঝ","›":"ট্ট","Uò":"ট্ব","Uô":"ট্ম","œ":"ড্ড","Ÿ":"ণ্ঠ","Ý":"ন্স","¡":"ণ্ড","š‘":"ন্তু","ìç":"ন্তু","ìÿ":"ন্থ","Yð":"ণ্ব","ª":"ন্ঠ","£ò":"ত্ত্ব","¤":"ত্থ","Zí":"ত্ন","£ô":"ত্ম","Zô":"ত্ম","£":"ত্ত","Zò":"ত্ব","¢":"ত্র","aò":"থ্ব","¥":"দ্দ","¦":"দ্ধ","§":"দ্ব","¨":"দ্ভ","bô":"দ্ম","cµ":"ধ্ব","ëU":"ন্ট","åU":"ন্ট","«":"ন্ড","ìæ":"ন্ত","ìòæ":"ন্ত্ব","ìè":"ন্ত্র","ëb":"ন্দ","ë§":"ন্দ্ব","¬":"ন্ধ","ëc":"ন্ধ","Ò":"ন্ন","dí":"ন্ন","ìñ":"ন্ব","dô":"ন্ম","ëo":"ন্স","ïU":"প্ট","®":"প্ত","eí":"প্ন","¯":"প্প","eö":"প্ল","d¬":"ফ্ল","fõ":"ফ্ল","±":"ব্জ","²":"ব্দ","³":"ব্ধ","gµ":"ব্ব","gö":"ব্ল","gõ":"ব্ল","»":"ম্ভ্র","ó¸":"ম্ভ্র","¸":"ভ্র","ií":"ম্ন","óe":"ম্প","óf":"ম্ফ","¹":"ম্ব","º":"ম্ভ","ói":"ম্ম","óö":"ম্ল","¿":"ল্ক","ùM":"ল্গ","ùU":"ল্ট","À":"ল্ড","ùe":"ল্প","ùf":"ল্ফ","lð":"ল্ব","lô":"ল্ম","Á":"ল্ল","lö":"ল্ল","lõ":"ল্ল","Â":"শু","úP":"শ্চ","mí":"শ্ন","mð":"শ্ব","mô":"শ্ম","mö":"শ্ল","ûK":"ষ্ক","û¢ß":"ষ্ক্র","Ä":"ষ্ট","Å":"ষ্ঠ","ûe":"ষ্প","üf":"ষ্ফ","ûô":"ষ্ম","Æ":"স্ক","ýL":"স্খ","ýU":"স্ট","þU":"স্ট","þÿ":"স্খ","þæ":"স্ত","þç":"স্তু","þè":"স্ত্র","þí":"স্ন","oí":"স্ন","þe":"স্প","ýf":"স্ফ","È":"স্ব","þñ":"স্ব","þô":"স্ম","É":"হু","pî":"হ্ণ","pß":"হ্ন","Ê":"হ্ম","n¬":"হ্ল","põ":"হ্ল","pÕ":"হৃ","ò":"ব","Aw":"আ","A":"অ","B":"ই","C":"ঈ","D":"উ","E":"ঊ","F":"ঋ","G":"এ","H":"ঐ","I":"ও","J":"ঔ","K":"ক","L":"খ","M":"গ","N":"ঘ","O":"ঙ","P":"চ","Q":"ছ","R":"জ","S":"ঝ","T":"ঞ","U":"ট","V":"ঠ","W":"ড","X":"ঢ","Y":"ণ","Z":"ত","¤":"থ","b":"দ","c":"ধ","d":"ন","e":"প","f":"ফ","g":"ব","h":"ভ","i":"ম","j":"য","k":"র","l":"ল","m":"শ","n":"ষ","o":"স","p":"হ","r":"ড়","s":"ঢ়","t":"য়","a":"থ","u":"ৎ","0":"০","1":"১","2":"২","3":"৩","4":"৪","5":"৫","6":"৬","7":"৭","8":"৮","9":"৯","w":"া","x":"ি","y":"ী","×":"ু","Ö":"ু","Ø":"ু","Ô":"ূ","Õ":"ূ","Ó":"ূ","Ù":"ৃ","Ú":"ৃ","Ë":"ে","Ì":"ে","Ð":"ৈ","Ñ":"ৈ","\\ˆ":"ৈ","#":"ৗ","„":"।","z":"্","v":"ং","\\^":"ঁ","Þ":"্র","Û":"্র","ê":"র্","ø":"্য"
};

// ---------------------- Re-Arrange function for ASCII->Unicode (from bijoy file) ----------------------

function ReArrangeUnicodeConvertedText(str) {
    for (var i=0;i<str.length;i++) {
        if (i>0 && str.charAt(i)=='\u09CD' && (IsBanglaKar(str.charAt(i-1))||IsBanglaNukta(str.charAt(i-1))) && i<str.length-1) {
            var temp=str.substring(0,i-1);
            temp+=str.charAt(i);
            temp+=str.charAt(i+1);
            temp+=str.charAt(i-1);
            temp+=str.substring(i+2,str.length);
            str=temp;
        }
        if(i>0 && i<str.length-1 && str.charAt(i)=='\u09CD' && str.charAt(i-1)=='\u09B0' && str.charAt(i-2)!='\u09CD' && IsBanglaKar(str.charAt(i+1))) {
            var temp=str.substring(0,i-1);
            temp+=str.charAt(i+1);
            temp+=str.charAt(i-1);
            temp+=str.charAt(i);
            temp+=str.substring(i+2,str.length);
            str=temp;
        }
        if(i<str.length-1 && str.charAt(i)=='র' && IsBanglaHalant(str.charAt(i+1)) && !IsBanglaHalant(str.charAt(i-1))) {
            var j=1;
            while(true) {
                if(i-j<0) break;
                if(IsBanglaBanjonborno(str.charAt(i-j)) && IsBanglaHalant(str.charAt(i-j-1))) j+=2;
                else if(j==1 && IsBanglaKar(str.charAt(i-j))) j++;
                else break;
            }
            var temp=str.substring(0,i-j);
            temp+=str.charAt(i);
            temp+=str.charAt(i+1);
            temp+=str.substring(i-j,i);
            temp+=str.substring(i+2,str.length);
            str=temp;
            i+=1;
            continue;
        }
        if(i<str.length-1 && IsBanglaPreKar(str.charAt(i)) && IsSpace(str.charAt(i+1))==false) {
            var temp=str.substring(0,i);
            var j=1;
            while(IsBanglaBanjonborno(str.charAt(i+j))) {
                if(IsBanglaHalant(str.charAt(i+j+1))) j+=2;
                else break;
            }
            temp+=str.substring(i+1,i+j+1);
            var l=0;
            if(str.charAt(i)=='ে' && str.charAt(i+j+1)=='া') { temp+="ো"; l=1; }
            else if(str.charAt(i)=='ে' && str.charAt(i+j+1)=="ৗ") { temp+="ৌ"; l=1; }
            else temp+=str.charAt(i);
            temp+=str.substring(i+j+l+1,str.length);
            str=temp;
            i+=j;
        }
        if(i<str.length-1 && str.charAt(i)=='ঁ' && IsBanglaPostKar(str.charAt(i+1))) {
            var temp=str.substring(0,i);
            temp+=str.charAt(i+1);
            temp+=str.charAt(i);
            temp+=str.substring(i+2,str.length);
            str=temp;
        }
    }
    return str;
}

// ---------------------- Unicode -> ASCII maps (from "unicode file") ----------------------

var uni2bijoy_string_conversion_map={"।":"|","‘":"Ô","’":"Õ","“":"Ò","”":"Ó","্র্য":"ª¨","র্য":"i¨","ক্ক":"°","ক্ট":"±","ক্ত":"³","ক্ব":"K¡","স্ক্র":"¯Œ","ক্র":"µ","ক্ল":"K¬","ক্ষ":"¶","ক্স":"•","গু":"¸","গ্ধ":"»","গ্ন":"Mœ","গ্ম":"M¥","গ্ল":"Mø","গ্রু":"MÖæ","ঙ্ক":"¼","ঙ্ক্ষ":"•¶","ঙ্খ":"•L","ঙ্গ":"½","ঙ্ঘ":"•N","চ্চ":"”P","চ্ছ":"”Q","চ্ছ্ব":"”Q¡","চ্ঞ":"”T","জ্জ্ব":"¾¡","জ্জ":"¾","জ্ঝ":"À","জ্ঞ":"Á","জ্ব":"R¡","ঞ্চ":"Â","ঞ্ছ":"Ã","ঞ্জ":"Ä","ঞ্ঝ":"Å","ট্ট":"Æ","ট্ব":"U¡","ট্ম":"U¥","ড্ড":"Ç","ণ্ট":"È","ণ্ঠ":"É","ন্স":"Ý","ণ্ড":"Ê","ন্তু":"š‘","ণ্ব":"Y^","ত্ত":"Ë","ত্ত্ব":"Ë¡","ত্থ":"Ì","ত্ন":"Zœ","ত্ম":"Z¥","ন্ত্ব":"š—¡","ত্ব":"Z¡","থ্ব":"_¡","দ্গ":"˜M","দ্ঘ":"˜N","দ্দ":"Ï","দ্ধ":"×","দ্ব":"˜¡","দ্ভ":"™¢","দ্ম":"Ù","দ্রু":"`ªæ","ধ্ব":"aŸ","ধ্ম":"a¥","ন্ট":"›U","ন্ঠ":"Ú","ন্ড":"Û","ন্ত্র":"š¿","ন্ত":"šÍ","স্ত্র":"¯¿","ত্র":"Î","ন্থ":"š’","ন্দ":"›`","ন্দ্ব":"›Ø","ন্ধ":"Ü","ন্ন":"bœ","ন্ব":"š^","ন্ম":"b¥","প্ট":"Þ","প্ত":"ß","প্ন":"cœ","প্প":"à","প্ল":"c&","প্স":"á","ফ্ল":"d¬","ব্জ":"â","ব্দ":"ã","ব্ধ":"ä","ব্ব":"eŸ","ব্ল":"eø","ভ্র":"å","ম্ন":"gœ","ম্প":"¤ú","ম্ফ":"ç","ম্ব":"¤^","ম্ভ":"¤¢","ম্ভ্র":"¤£","ম্ম":"¤§","ম্ল":"¤ø","রু":"iæ","রূ":"iƒ","ল্ক":"é","ল্গ":"ê","ল্প":"í","ল্ট":"ë","ল্ড":"ì","ল্ফ":"î","ল্ব":"j¦","ল্ম":"j¥","ল্ল":"jø","শু":"ï","শ্চ":"ð","শ্ন":"kœ","শ্ব":"k¦","শ্ম":"k¥","শ্ল":"kø","ষ্ক":"®‹","ষ্ক্র":"®Œ","ষ্ট":"ó","ষ্ঠ":"ô","ষ্ণ":"ò","ষ্প":"®ú","ষ্ফ":"õ","ষ্ম":"®§","স্ক":"¯‹","স্ট":"÷","স্খ":"ö","স্ত":"¯—","স্তু":"¯‘","স্থ":"¯’","স্ন":"mœ","স্প":"¯ú","স্ফ":"ù","স্ব":"¯^","স্ম":"¯§","স্ল":"lø","হু":"û","হ্ণ":"nè","হ্ন":"ý","হ্ম":"þ","হ্ল":"n¬","হৃ":"ü","র্":"©","্র":"Ö","্য":"¨","্":"&","আ":"Av","অ":"A","ই":"B","ঈ":"C","উ":"D","ঊ":"E","ঋ":"F","এ":"G","ঐ":"H","ও":"I","ঔ":"J","ক":"K","খ":"L","গ":"M","ঘ":"N","ঙ":"O","চ":"P","ছ":"Q","জ":"R","ঝ":"S","ঞ":"T","ট":"U","ঠ":"V","ড":"W","ঢ":"X","ণ":"Y","ত":"Z","থ":"_","দ":"`","ধ":"a","ন":"b","প":"c","ফ":"d","ব":"e","ভ":"f","ম":"g","য":"h","র":"i","ল":"j","শ":"k","ষ":"l","স":"m","হ":"n","ড়":"o","ঢ়":"p","য়":"q","ৎ":"r","০":"0","১":"1","২":"2","৩":"3","৪":"4","৫":"5","৬":"6","৭":"7","৮":"8","৯":"9","া":"v","ি":"w","ী":"x","ু":"y","ূ":"~","ৃ":"…","ে":"‡","ৈ":"‰","ৗ":"Š","ং":"s","ঃ":"t","ঁ":"u","—": "-"};
var uni2somewherein_string_conversion_map={"।":"|","‘":"Ô","’":"Õ","“":"Ò","”":"Ó","্র্য":"ª¨","র‌্য":"i¨","ক্ক":"°","ক্ট":"±","ক্ত":"³","ক্ব":"K¡","স্ক্র":"¯Œ","ক্র":"µ","ক্ল":"K¬","ক্ষ":"¶","ক্স":"·","গু":"¸","গ্ধ":"»","গ্ন":"Mœ","গ্ম":"M¥","গ্ল":"M­","গ্রু":"Mªy","ঙ্ক":"¼","ঙ্ক্ষ":"•¶","ঙ্খ":"•L","ঙ্গ":"½","ঙ্ঘ":"•N","চ্চ":"”P","চ্ছ":"”Q","চ্ছ্ব":"”Q¡","চ্ঞ":"”T","জ্জ্ব":"¾¡","জ্জ":"¾","জ্ঝ":"À","জ্ঞ":"Á","জ্ব":"R¡","ঞ্চ":"Â","ঞ্ছ":"Ã","ঞ্জ":"Ä","ঞ্ঝ":"Å","ট্ট":"Æ","ট্ব":"U¡","ট্ম":"U¥","ড্ড":"Ç","ণ্ট":"È","ণ্ঠ":"É","ন্স":"Ý","ণ্ড":"Ê","ন্তু":"š‘","ণ্ব":"Y^","ত্ত":"Ë","ত্ত্ব":"Ë¡","ত্থ":"Ì","ত্ন":"Zœ","ত্ম":"Z¥","ন্ত্ব":"š—¡","ত্ব":"Z¡","থ্ব":"_¡","দ্গ":"˜M","দ্ঘ":"˜N","দ্দ":"Ï","দ্ধ":"×","দ্ব":"˜¡","দ্ভ":"™¢","দ্ম":"Ù","দ্রু":"`ª“","ধ্ব":"aŸ","ধ্ম":"a¥","ন্ট":"›U","ন্ড":"Û","ন্ত্র":"š¿","ন্ত":"š—","স্ত্র":"¯¿","ত্র":"Î","ন্থ":"š’","ন্দ":"›`","ন্দ্ব":"›Ø","ন্ধ":"Ü","ন্ন":"bœ","ন্ব":"š^","ন্ম":"b¥","প্ট":"Þ","প্ত":"ß","প্ন":"cœ","প্প":"à","প্ল":"c­","প্স":"á","ফ্ল":"d¬","ব্জ":"â","ব্দ":"ã","ব্ধ":"ä","ব্ব":"eŸ","ব্ল":"e­","ভ্র":"å","ম্ন":"gœ","ম্প":"¤ú","ম্ফ":"ç","ম্ব":"¤^","ম্ভ":"¤¢","ম্ভ্র":"¤£","ম্ম":"¤§","ম্ল":"¤­","রু":"i“","রূ":"iƒ","ল্ক":"é","ল্গ":"ê","ল্প":"í","ল্ট":"ë","ল্ড":"ì","ল্ফ":"î","ল্ব":"j¦","ল্ম":"j¥","ল্ল":"j­","শু":"ï","শ্চ":"ð","শ্ন":"kœ","শ্ব":"k¦","শ্ম":"k¥","শ্ল":"k­","ষ্ক":"®‹","ষ্ক্র":"®Œ","ষ্ট":"ó","ষ্ঠ":"ô","ষ্ণ":"ò","ষ্প":"®ú","ষ্ফ":"õ","ষ্ম":"®§","স্ক":"¯‹","স্ট":"÷","স্খ":"ö","স্ত":"¯—","স্তু":"¯‘","স্থ":"¯’","স্ন":"mœ","স্প":"¯ú","স্ফ":"ù","স্ব":"¯^","স্ম":"¯§","স্ল":"¯­","হ্ণ":"nè","হ্ন":"ý","হ্ম":"þ","হু":"û","হ্ল":"n¬","হৃ":"ü","র্":"©","্র":"Ö","্য":"¨","্":"&","আ":"Av","অ":"A","ই":"B","ঈ":"C","উ":"D","ঊ":"E","ঋ":"F","এ":"G","ঐ":"H","ও":"I","ঔ":"J","ক":"K","খ":"L","গ":"M","ঘ":"N","ঙ":"O","চ":"P","ছ":"Q","জ":"R","ঝ":"S","ঞ":"T","ট":"U","ঠ":"V","ড":"W","ঢ":"X","ণ":"Y","ত":"Z","থ":"_","দ":"`","ধ":"a","ন":"b","প":"c","ফ":"d","ব":"e","ভ":"f","ম":"g","য":"h","র":"i","ল":"j","শ":"k","ষ":"l","স":"m","হ":"n","ড়":"o","ঢ়":"p","য়":"q","ৎ":"r","০":"0","১":"1","২":"2","৩":"3","৪":"4","৫":"5","৬":"6","৭":"7","৮":"8","৯":"9","া":"v","ি":"w","ী":"x","ু":"y","ূ":"~","ৃ":"„","ে":"‡","ৈ":"‰","ৗ":"Š","ং":"s","ঃ":"t","ঁ":"u"};
var uni2boisakhi_string_conversion_map={"্র্য":"Ûø","র‌্য":"kø","ক্ক":"~","ক্ট":"ƒ","ক্ত":"£ß","ক্ব":"Kò","স্ক্র":"Ç","ক্র":"¢ß","ক্ষ্ম":"qô","ক্ষ":"q","ক্স":"…","ক্ল":"Kõ","গু":"†","গ্গ":"‡","গ্ধ":"ˆ","গ্ন":"Mí","গ্ম":"Mô","গ্ল":"Mö","ঙ্ক":"‰","ঙ্ক্ষ":"áq","ঙ্খ":"áL","ঙ্গ":"Š","ঙ্ঘ":"áN","চ্চ":"âP","চ্ছ":"âQ","চ্ছ্ব":"âQò","জ্জ্ব":"Œò","জ্জ":"Œ","জ্ঞ":"š","জ্ব":"Rò","ঞ্চ":"é","˜":"ঞ্ছ","ঞ্জ":"™","ঞ্ঝ":"ã","ট্ট":"›","ট্ব":"Uò","ট্ম":"Uô","ড্ড":"œ","ণ্ঠ":"Ÿ","ন্ঠ":"ª","ন্স":"Ý","ণ্ড":"¡","ন্তু":"š‘","ণ্ব":"Yð","ত্ত্ব":"£ò","ত্থ":"¤","ত্ন":"Zí","ত্ম":"£ô","ত্ম":"Zô","ত্ত":"£","ন্ত্ব":"ìòæ","ত্ব":"Zò","থ্ব":"aò","দ্দ":"¥","দ্ধ":"¦","দ্ব":"§","দ্ভ":"¨","দ্ম":"bô","ধ্ব":"cµ","ন্ট":"ëU","ন্ট":"åU","ন্ড":"«","ন্ত্র":"ìè","ন্ত":"ìæ","স্ত্র":"þè","ত্র":"¢","ন্দ":"ëb","ন্দ্ব":"ë§","ন্ধ":"¬","ন্ন":"Ò","ন্ব":"ìñ","ন্থ":"ìÿ","ন্ম":"dô","প্ট":"ïU","প্ত":"®","প্ন":"eí","প্প":"¯","প্ল":"eö","ফ্ল":"d¬","ব্জ":"±","ব্দ":"²","ব্ধ":"³","ব্ব":"gµ","ব্ল":"gö","ম্ভ্র":"»","ভ্র":"¸","ম্ন":"ií","ম্প":"óe","ম্ফ":"óf","ম্ব":"¹","ম্ভ":"º","ম্ম":"ói","ম্ল":"óö","ল্ক":"¿","ল্গ":"ùM","ল্ট":"ùU","ল্ড":"À","ল্প":"ùe","ল্ফ":"ùf","ল্ব":"lð","ল্ম":"lô","ল্ল":"Á","শু":"Â","শ্চ":"úP","শ্ন":"mí","শ্ব":"mð","শ্ম":"mô","শ্ল":"mö","ষ্ক":"ûK","ষ্ক্র":"û¢ß","ষ্ট":"Ä","ষ্ঠ":"Å","ষ্প":"ûe","ষ্ফ":"üf","ষ্ম":"ûô","স্ক":"Æ","স্খ":"ýL","স্ট":"ýU","স্থ":"þÿ","স্ত":"þæ","স্তু":"þç","স্ন":"þí","স্প":"þe","স্ফ":"ýf","স্ব":"È","স্ব":"þñ","স্ম":"þô","হু":"É","হ্ণ":"pî","হ্ন":"pß","হ্ম":"Ê","হ্ল":"n¬","হ্ল":"põ","হৃ":"pÕ","ব":"ò","র্":"ê","্র":"Þ","্য":"ø","্":"z","আ":"Aw","অ":"A","ই":"B","ঈ":"C","উ":"D","ঊ":"E","ঋ":"F","এ":"G","ঐ":"H","ও":"I","ঔ":"J","ক":"K","খ":"L","গ":"M","ঘ":"N","ঙ":"O","চ":"P","ছ":"Q","জ":"R","ঝ":"S","ঞ":"T","ট":"U","ঠ":"V","ড":"W","ঢ":"X","ণ":"Y","ত":"Z","থ":"¤","দ":"b","ধ":"c","ন":"d","প":"e","ফ":"f","ব":"g","ভ":"h","ম":"i","য":"j","র":"k","ল":"l","শ":"m","ষ":"n","স":"o","হ":"p","ড়":"r","ঢ়":"s","য়":"t","থ":"a","ৎ":"u","ং":"v","ঁ":"^","০":"0","১":"1","২":"2","৩":"3","৪":"4","৫":"5","৬":"6","৭":"7","৮":"8","৯":"9","া":"w","ি":"x","ী":"y","ু":"×","ূ":"Ô","ৃ":"Ú","ে":"Ë","ৈ":"Ð","ৗ":"#","।":"„"};

// ---------------------- Re-Arrange function for Unicode->ASCII (from unicode file) ----------------------

function ReArrangeUnicodeText(str) {
    var barrier=0;
    for (var i=0;i<str.length;i++) {
        if (i<str.length && IsBanglaPreKar(str.charAt(i))) {
            var j=1;
            while(IsBanglaBanjonborno(str.charAt(i-j))) {
                if (i-j<0) break;
                if (i-j<=barrier) break;
                if (IsBanglaHalant(str.charAt(i-j-1))) j+=2; else break;
            }
            var temp=str.substring(0,i-j);
            temp+=str.charAt(i);
            temp+=str.substring(i-j,i);
            temp+=str.substring(i+1,str.length);
            str=temp;
            barrier=i+1;
            continue;
        }
        if (i < str.length-1 && IsBanglaHalant(str.charAt(i)) && str.charAt(i-1)=='র' && !IsBanglaHalant(str.charAt(i-2))) {
            var j=1; var found_pre_kar=0;
            while(true) {
                if (IsBanglaBanjonborno(str.charAt(i+j)) && IsBanglaHalant(str.charAt(i+j+1))) j+=2;
                else if (IsBanglaBanjonborno(str.charAt(i+j)) && IsBanglaPreKar(str.charAt(i+j+1))) { found_pre_kar=1; break; }
                else break;
            }
            var temp=str.substring(0,i-1);
            temp+=str.substring(i+j+1,i+j+found_pre_kar+1);
            temp+=str.substring(i+1,i+j+1);
            temp+=str.charAt(i-1);
            temp+=str.charAt(i);
            temp+=str.substring(i+j+found_pre_kar+1,str.length);
            str=temp;
            i+=(j+found_pre_kar);
            barrier=i+1;
            continue;
        }
    }
    return str;
}

// ---------------------- Convert functions (two-way) ----------------------

// Unicode -> ASCII (bijoy/somewherein/boisakhi)
function ConvertToASCII(ConvertTo, line) {
    var conversion_map = uni2bijoy_string_conversion_map;
    if (ConvertTo == "bijoy") conversion_map = uni2bijoy_string_conversion_map;
    else if (ConvertTo == "somewherein") conversion_map = uni2somewherein_string_conversion_map;
    else if (ConvertTo == "boisakhi") conversion_map = uni2boisakhi_string_conversion_map;

    var myRegExp;
    myRegExp = new RegExp("ো","g"); line = line.replace(myRegExp,"ো");
    myRegExp = new RegExp("ৌ","g"); line = line.replace(myRegExp,"ৌ");
    line = ReArrangeUnicodeText(line);

    for (var unic in conversion_map) {
        myRegExp = new RegExp(unic,"g");
        line = line.replace(myRegExp, conversion_map[unic]);
    }
    return line;
}

// ASCII -> Unicode (bijoy/somewherein/boisakhi)
function ConvertToUnicode(ConvertFrom, line) {
    var conversion_map = bijoy_string_conversion_map;
    if (ConvertFrom == "bijoy") conversion_map = bijoy_string_conversion_map;
    else if (ConvertFrom == "somewherein") conversion_map = somewherein_string_conversion_map;
    else if (ConvertFrom == "boisakhi") conversion_map = boisakhi_string_conversion_map;

    for (var ascii in conversion_map) {
        var myRegExp = new RegExp(ascii,"g");
        line = line.replace(myRegExp, conversion_map[ascii]);
    }
    // rearrange (as bijoy file did) and small normalization
    line = ReArrangeUnicodeConvertedText(line);
    var myRegExp = new RegExp("অা","g"); line = line.replace(myRegExp,"আ");
    return line;
}

// ---------------------- Exports / global bindings (optional) ----------------------
// If you use modules, you may export functions. For browser, attach to window:
if (typeof window !== "undefined") {
    window.BanglaConverter = {
        ConvertToASCII: ConvertToASCII,
        ConvertToUnicode: ConvertToUnicode,
        ReArrangeUnicodeText: ReArrangeUnicodeText,
        ReArrangeUnicodeConvertedText: ReArrangeUnicodeConvertedText
    };
}

// ---------------------- End of file ----------------------
